' Matrix Inversion using Gauss-Jordan Elimination
' Will Invert a square matrix up to SIZE\2.
' Feel free to increase SIZE if you need to.
' Note: Because of floating-point rounding, there will
' be very small errors in the inverted matrix. These are
' generally in the order of 10**-15 or smaller and can
' be safely ignored.
' This test program accepts values for a square matrix up
' to 5x5, calculates and displays its inverse, computes
' the product of the original matrix and the inverse, which
' will be the identity matrix with some small residuals
' off the diagonal. The matrix is tested for singularity
' by computing its determinant before the inverse is computed.
' You can extend this to solving systems of linear equations
' by multiplying the inverse matrix by the column vector of
' constants in the right-hand side of the equations in standard
' format.
' WARNINGS:
' The Gauss-Jordan Elimination method can be numerically 
' unstable for ill-conditioned matrices. If the determinant is
' very small but not zero, the errors may blow up into large
' values. Other pathological matrices can cause bad results.
' The algorithm's computational complexity is O(n**3), so
' it will be slow for high-order matrices.
' Rev 1.0.0 William M Leue

const SIZE = 10

dim float a(SIZE, SIZE), b(SIZE, SIZE)
dim float d
dim integer n

' Main Program
cls
input "Enter order of matrix: ", n
GetMatrixData n
CopyMatrix n, a(), b()
print ""
print "Original Matrix"
PrintMatrix n, a()
Determinant n, a(), d
if d = 0.0 then
  print "Matrix is Singular, cannot compute its inverse"
  end
else 
  print "Determinant: " + FORMAT$(d, "%+.3f")
end if
InvertMatrix n, a()
print ""
print "Inverted Matrix"
printMatrix n, a()
TestInversion n, b(), a()
end

' Get the matrix coefficients from the user
sub GetMatrixData n
  if n > SIZE\2 then
    print "Too large: n must be <= 5"
    end
  end if
  print "Enter coefficients of matrix, one row at a time:"

  for i = 1 to n
    print "  row " + str$(i) + ":"
    for j i = 1 to n
      print "a(" + str$(i) + "," + str$(j) + "): ";
      input "", a(i, j)
    next j
  next i
end sub

' Copy one matrix to another
sub CopyMatrix n, s(), d()
  local integer i, j
  for i = 1 to n
    for j = 1 to n
      d(i, j) = s(i, j)
    next j
  next i
end sub

' Print a matrix in a user-friendly format.
' Values are rounded off to 3 digits after
' the decimal point. Change the format if you
' need more precision.
sub PrintMatrix n, m()
  local integer i, j
  local v$
  print ""
  for i = 1 to n
    for j = 1 to n
      v$ = FORMAT$(m(i, j), "%+.3f")
      print v$ + " ";
    next j
    print ""
  next i
end sub

' Invert a Matrix using Gauss-Jordan Elimination.
' The inverse replaces the values in the input matrix,
' so copy those to another matrix before calling this
' subroutine if you need them later on.
sub InvertMatrix n, m()
  local i, j, k

  'Augment input matrix by identity matrix order N
  for i = 1 to n
    for j = 1 to n
      if i = j then
        a(i, j+n) = 1
      else
        a(i, j+n) = 0
      end if
    next j
  next i
  ' Apply Gauss-Jorden Elimination
  for i = 1 to n
    if a(i, i) = 0.0 then
      print "Math Error: zero diagonal element!"
      end
    end if
    for j = 1 to n
      if i <> j then
        ratio = a(j, i)/a(i, i)
        for k = 1 to 2*n
          a(j, k) = a(j, k) - ratio*a(i, k)
        next k
      end if
    next j
  next i
  ' Apply row operation to make principal diagonal = 1
  for i = 1 to n
    for j = n+1 to 2*n
      a(i, j) = a(i, j)/a(i, i)
    next j
  next i
  ' Copy extended matrix back to original coordinates
  for i = 1 to n
    for j = 1 to n
      a(i, j) = a(i, j+n)
    next j
  next i
end sub

' Compute and return the determinant of the supplied
' matrix of order n.
sub Determinant n as integer, m() as float, d as float
  local float minor(2*SIZE, 2*SIZE)
  local integer i, j, k, c1, c2, sign
  if n = 2 then
    d = m(1, 1)*m(2, 2) - m(1, 2)*m(2, 1)
    exit sub
  end if
  d = 0.0
  c1 = 1 : c2 = 1
  sign = 1
  for i = 1 to n
    for j = 1 to n
      for k = 1 to n
        if j <> 1 and k <> i then
          minor(c1, c2) = m(j, k)
          inc c2
          if c2 > n-3 then
            inc c1
            c2 = 1
          end if
        end if
      next k
    next j
    inc d, sign*m(1, i)
    sign = -1*sign
  next i
end sub

' Test the Inversion by Multiplying the Original by the Inverse
' Should come out to the identity matrix with perhaps some small
' Error terms due to floating-point roundoff.
sub TestInversion n, orig() as float, inverse() as float
  local float prod(SIZE, SIZE)
  MATH M_MULT orig(), inverse(), prod()
  print ""
  print "product of original matrix and inverse"
  PrintMatrix n, prod()
end sub  
